-- Mazebot Project
-- EE552 Fall 2000
--
-- Authors:
-- Steve Dillen 225760
--
-- October 9, 2000
--
-- Motor_control.vhd
-- This file implements a motor controller for 2 continuous DC motor's
-- It takes as input 2 n bit number indicating the velocity of each
-- motor.  The controller will try to control each velocity to a set velocity
-- passed in as the reference velocity.  The starting PWM should also
-- be passed in.
--
-- This will output a PWM signal with the appropriate duty cycle to achieve 
-- the desired velocity for each wheel.
--

-- Include ieee std logic types
--
library ieee;
use ieee.std_logic_1164.all;

-- Include Mazebot configuration
--
library work;
use work.config.all;

entity motor_control is

  generic ( busWidth         : positive := 4; 

            -- Due to a compile bug with MaxPlus2, this must be specified to be
            -- busWidth + 1.
            --
            extendedBusWidth : positive := 5;
            numberOfMotors   : positive := 2;
            fuzzyInWidth     : positive := 3;
            fuzzyOutWidth    : positive := 3
          );

  port    ( clock            : in  std_logic;
            reset            : in  std_logic;
            velocity         : in  std_logic_vector( numberOfMotors * busWidth - 1 downto 0 );
            reference        : in  std_logic_vector( numberOfMotors * busWidth - 1 downto 0 );
            initial_rate     : in  std_logic_vector( numberOfMotors * busWidth - 1 downto 0 );
            diff_membership  : in  std_logic_vector( 7 * fuzzyInWidth - 1 downto 0 );
            int_membership   : in  std_logic_vector( 7 * fuzzyInWidth - 1 downto 0 );
            pwm_membership   : in  std_logic_vector( 7 * fuzzyOutWidth - 1 downto 0 );
            valid            : in  std_logic_vector( numberOfMotors - 1 downto 0 );
            pwm_signal       : out std_logic_vector( numberOfMotors - 1 downto 0 );
            fuzzy_out        : out std_logic_vector( 2 downto 0 );
            fuzzy_in         : out std_logic_vector( 2 downto 0 )
          );

end motor_control;

architecture RTL of motor_control is

  component controller is

    generic ( busWidth         : positive;

              -- Due to a compile bug with MaxPlus2, this must be specified to be
              -- busWidth + 1.
              --
              extendedBusWidth : positive;

              fuzzyInWidth     : positive;
              fuzzyOutWidth    : positive
            );

    port    ( clock            : in  std_logic;
              reset            : in  std_logic;
              data_a           : in  std_logic_vector( busWidth - 1 downto 0 );
              data_b           : in  std_logic_vector( busWidth - 1 downto 0 );
              reference        : in  std_logic_vector( busWidth - 1 downto 0 );
              diff_membership  : in  std_logic_vector( 7 * fuzzyInWidth - 1 downto 0 );
              int_membership   : in  std_logic_vector( 7 * fuzzyInWidth - 1 downto 0 );
              pwm_membership   : in  std_logic_vector( 7 * fuzzyOutWidth - 1 downto 0 );
              valid_in         : in  std_logic;
              valid_out        : out std_logic;
              control          : out std_logic_vector( busWidth - 1 downto 0 );
              fuzzy_out        : out std_logic_vector( 2 downto 0 );
              fuzzy_in         : out std_logic_vector( 2 downto 0 )
            );        

  end component controller;

  component duty_rate_generator is

    generic ( busWidth         : positive;

              -- Due to a compile bug with MaxPlus2, this must be specified to be
              -- busWidth + 1.
              --
              extendedBusWidth : positive
            );

    port    ( clock        : in  std_logic;
              reset        : in  std_logic;
              adjustment   : in  std_logic_vector( busWidth - 1 downto 0 );
              initial_rate : in  std_logic_vector( busWidth - 1 downto 0 );
              valid_in     : in  std_logic;
              duty_rate    : out std_logic_vector( busWidth - 1 downto 0 )
            );

  end component duty_rate_generator;

  component pwm_generator is

    generic ( busWidth   : positive );

    port    ( clock      : in  std_logic;
              reset      : in  std_logic;
              duty_cycle : in  std_logic_vector( busWidth - 1 downto 0 );
              pwm_signal : out std_logic
            );

  end component pwm_generator;

  -- Signals to connect the controller to the duty rate generator
  --
  signal internal_adjustment_valid : std_logic_vector( numberOfMotors - 1 downto 0 );
  signal internal_adjustment       : std_logic_vector( numberOfMotors * busWidth - 1 downto 0 );

  -- Signals to connect the duty rate generator to the pwm generator
  --
  signal internal_duty_rate        : std_logic_vector( numberOfMotors * busWidth - 1 downto 0 );

  signal internal_fuzzy_out        : std_logic_vector( numberOfMotors * 3 - 1 downto 0 );
  signal internal_fuzzy_in         : std_logic_vector( numberOfMotors * 3 - 1 downto 0 );

begin

  fuzzy_out <= internal_fuzzy_out( 2 downto 0 );
  fuzzy_in  <= internal_fuzzy_in( 2 downto 0 );

  fuzzy_0 : controller generic map (
                                     busWidth         => busWidth,
                                     extendedBusWidth => extendedBusWidth,
                                     fuzzyInWidth     => fuzzyInWidth,
                                     fuzzyOutWidth    => fuzzyOutWidth
                                   )
                       port    map (
                                     clock            => clock,
                                     reset            => reset,
                                     data_a           => velocity( busWidth * 1 - 1 downto busWidth * 0 ),
                                     data_b           => velocity( busWidth * 2 - 1 downto busWidth * 1 ),
                                     reference        => reference( busWidth * 1 - 1 downto busWidth * 0 ),
                                     diff_membership  => diff_membership,
                                     int_membership   => int_membership,
                                     pwm_membership   => pwm_membership,
                                     valid_in         => valid(0),
                                     valid_out        => internal_adjustment_valid(0),
                                     control          => internal_adjustment( busWidth * 1 - 1 downto busWidth * 0 ),
                                     fuzzy_out        => internal_fuzzy_out( 3 * 1 - 1 downto 3 * 0 ),
                                     fuzzy_in         => internal_fuzzy_in( 3 * 1 - 1 downto 3 * 0 )
                                   );

  fuzzy_1 : controller generic map (
                                     busWidth         => busWidth,
                                     extendedBusWidth => extendedBusWidth,
                                     fuzzyInWidth     => fuzzyInWidth,
                                     fuzzyOutWidth    => fuzzyOutWidth
                                   )
                       port    map (
                                     clock            => clock,
                                     reset            => reset,
                                     data_a           => velocity( busWidth * 2 - 1 downto busWidth * 1 ),
                                     data_b           => velocity( busWidth * 1 - 1 downto busWidth * 0 ),
                                     reference        => reference( busWidth * 2 - 1 downto busWidth * 1 ),
                                     diff_membership  => diff_membership,
                                     int_membership   => int_membership,
                                     pwm_membership   => pwm_membership,
                                     valid_in         => valid(1),
                                     valid_out        => internal_adjustment_valid(1),
                                     control          => internal_adjustment( busWidth * 2 - 1 downto busWidth * 1 ),
                                     fuzzy_out        => internal_fuzzy_out( 3 * 2 - 1 downto 3 * 1 ),
                                     fuzzy_in         => internal_fuzzy_in( 3 * 2 - 1 downto 3 * 1 )
                                   );

  -- Generate a controller, a duty rate generator, and a
  -- PWM generator for each each motor.
  --
  control : for i in 0 to numberOfMotors - 1 generate

    duty_rate_i : duty_rate_generator generic map ( 
                                                    busWidth         => busWidth,
                                                    extendedBusWidth => extendedBusWidth 
                                                  )
                                      port    map (
                                                    clock            => clock,
                                                    reset            => reset,
                                                    adjustment       => internal_adjustment( busWidth * ( i + 1 ) - 1 downto busWidth * i ),
                                                    initial_rate     => initial_rate( busWidth * ( i + 1 ) - 1 downto busWidth * i ),
                                                    valid_in         => internal_adjustment_valid(i),
                                                    duty_rate        => internal_duty_rate( busWidth * ( i + 1 ) - 1 downto busWidth * i )
                                                  );

    -- Generate the PWM signals
    --
    pwm_i : pwm_generator generic map ( busWidth   => busWidth )
                          port    map (
                                        clock      => clock,
                                        reset      => reset,
                                        duty_cycle => internal_duty_rate( busWidth * ( i + 1 ) - 1 downto busWidth * i ),
                                        pwm_signal => pwm_signal( i )
                                      );

  end generate control;

end RTL;
