-- Mazebot Project
-- EE552 Fall 2000
--
-- Authors:
-- Steve Dillen 225760
--
-- November 13, 2000
--
-- duty_rate_generator.vhd
-- This module generates the duty rate for the PWM based on the adjustment
--

-- Include ieee std types
--
library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;

-- Include Mazebot configuration
--
library work;
use work.config.all;

entity duty_rate_generator is

  generic ( busWidth         : positive := 4;

            -- This is required since MaxPlus2 doesn't allow me to use
            -- busWidth and just add one to it, so this is the work around.
            -- It must always be specified as busWidth + 1.
            --
            extendedBusWidth : positive := 5
          );

  port    ( clock        : in  std_logic;
            reset        : in  std_logic;
            adjustment   : in  std_logic_vector( busWidth - 1 downto 0 );
            initial_rate : in  std_logic_vector( busWidth - 1 downto 0 );
            valid_in     : in  std_logic;
            duty_rate    : out std_logic_vector( busWidth - 1 downto 0 )
          );

end duty_rate_generator;

architecture RTL of duty_rate_generator is

  component latch_rate is

    generic ( busWidth   : positive );

    port    ( clock      : in  std_logic;
              reset      : in  std_logic;
              d          : in  std_logic_vector( busWidth - 1 downto 0 );
              initial    : in  std_logic_vector( busWidth - 2 downto 0 );
              valid_in   : in  std_logic;
              q          : out std_logic_vector( busWidth - 1 downto 0 )
            );

  end component latch_rate;

  signal adjusted_duty_rate  : std_logic_vector( busWidth downto 0 );
  signal current_duty_rate   : std_logic_vector( busWidth downto 0 );
  signal internal_valid      : std_logic;

  -- Signals used to sign extend the adjustment and current duty rate
  --
  signal extended_adjustment : std_logic_vector( busWidth downto 0 );

begin

  -- Adjust the duty rate
  --
  adjust : lpm_add_sub generic map ( 
                                     lpm_width          => extendedBusWidth,
                                     lpm_representation => "SIGNED",
                                     lpm_pipeline       => 1
                                   )
                       port    map (
                                     dataa              => extended_adjustment,
                                     datab              => adjusted_duty_rate,
                                     aclr               => reset,
                                     clock              => clock,
                                     clken              => valid_in,
                                     result             => adjusted_duty_rate
                                   );

  -- Synchronize the valid signal to stay at the same clock edge as the
  -- duty rate that will be latched.
  --
  sync_valid : synchronizor generic map ( numberOfLevels => 1 )
                            port    map ( 
                                          clock          => clock,
                                          reset          => reset,
                                          input          => valid_in,
                                          output         => internal_valid
                                        );
  
  -- Latch the rate, this will reset to the base rate
  --
  duty : latch_rate generic map ( busWidth => extendedBusWidth )
                    port    map (
                                  clock    => clock,
                                  reset    => reset,
                                  d        => adjusted_duty_rate,
                                  initial  => initial_rate,
                                  valid_in => internal_valid,
                                  q        => current_duty_rate
                                );

  -- Sign extend the adjustment
  extended_adjustment( busWidth - 1 downto 0 ) <= adjustment;
  extended_adjustment( busWidth )              <= adjustment( busWidth - 1 );

  -- Remove the leading positive bit from the duty rate
  --
  duty_rate <= current_duty_rate( busWidth - 1 downto 0 );

end RTL;
