-- Control-bot Project
-- EE552 Fall 2000
--
-- Authors:
-- Steve Dillen 225760
-- Farrah Rashid 354698
--
-- October 9, 2000
--
-- data_fusion.vhd
-- This file implements a data fusion engine that takes the sensor input
-- and output and hides the details from the other components
--

-- Include the IEEE std logic types
--
library ieee;
use ieee.std_logic_1164.all;

entity data_fusion is

  generic( busWidth       : positive := 4;
           numberOfMotors : positive := 2 
         );

  port ( 
         -- Control signals
         --
         clock            : in  std_logic;
         reset            : in  std_logic;
         shaft_resolution : in  std_logic;

         -- Motor signals
         --
         pwm_signal       : in  std_logic_vector( numberOfMotors - 1 downto 0 );
         direction        : in  std_logic_vector( numberOfMotors - 1 downto 0 );
         stop             : in  std_logic_vector( numberOfMotors - 1 downto 0 );
         forward          : out std_logic_vector( numberOfMotors - 1 downto 0 );
         reverse          : out std_logic_vector( numberOfMotors - 1 downto 0 );
         enable           : out std_logic_vector( numberOfMotors - 1 downto 0 );
 
         -- Shaft encoder signals
         --
         reflection       : in  std_logic_vector( numberOfMotors - 1 downto 0 );
         velocity_valid   : out std_logic_vector( numberOfMotors - 1 downto 0 );
         velocity         : out std_logic_vector( numberOfMotors * busWidth - 1 downto 0 )
       );

end data_fusion;

-- Start up the engine
--
architecture engine of data_fusion is

  -- Use 1 shaft encoder per motor.
  --
  component shaft_encoder

    generic ( busWidth       : positive );

    port    ( clock          : in  std_logic;
              reset          : in  std_logic;
              resolution     : in  std_logic;
              reflection     : in  std_logic;
              velocity_valid : out std_logic;
              velocity       : out std_logic_vector( busWidth - 1 downto 0 )
            );

  end component shaft_encoder;

  -- Use 1 motor_driver per motor
  --
  component motor_driver

    port( clock      : in  std_logic;
          reset      : in  std_logic;
          direction  : in  std_logic;
          pwm_signal : in  std_logic;
          stop       : in  std_logic;
          forward    : out std_logic;
          reverse    : out std_logic;
          enable     : out std_logic
        );

  end component motor_driver;

  signal one_signal           : std_logic;
  signal internal_wall_exists : std_logic_vector( 5 downto 0 );

begin

  -- Set to a one so we are constantly checking for walls
  --
  one_signal <= '1';

  -- Generate the motor's
  --
  motion : for i in 0 to numberOfMotors - 1 generate

    -- Declare variables to be used to index into the large distance
    -- array.
    constant startpt : positive := i * busWidth;
    constant endpt   : positive := ( ( i + 1 ) * busWidth ) - 1;

  begin
  
    motors : motor_driver port map (
                                     clock      => clock,
                                     reset      => reset,
                                     direction  => direction(i),
                                     pwm_signal => pwm_signal(i),
                                     stop       => stop(i),
                                     forward    => forward(i),
                                     reverse    => reverse(i),
                                     enable     => enable(i)
                                   );

    vel : shaft_encoder generic map ( busWidth       => busWidth )
                        port    map (
                                      clock          => clock,
                                      reset          => reset,
                                      resolution     => shaft_resolution,
                                      reflection     => reflection(i),
                                      velocity_valid => velocity_valid(i),
                                      velocity       => velocity( endpt downto startpt )
                                    );

  end generate motion;

end engine;
