-- Controlbot Project
--
-- Fall 2000
--
-- Group Members:
--
-- Steve Dillen  225760
-- Farrah Rashid 
--
-- File Author:
-- Steve Dillen  225760
--
-- This program instantiates all the external pins required
-- for Controlbot.  It then starts a shaft encoder for each motor, and 
-- a motor controller module to control the motors to a specified velocity.
--

-- Include IEEE standard types
--
library ieee;
use ieee.std_logic_1164.all;

-- Include Controlbot configuration
--
library work;
use work.config.all;

entity controlbot is

  port( 
        -- Control signals
        --
        clock            : in  std_logic;
        reset_signal     : in  std_logic;

        -- Shaft encoder signals
        --
        reflection       : in  std_logic_vector( numberOfMotors - 1 downto 0 );

        -- Motor signals
        --
        forward          : out std_logic_vector( numberOfMotors - 1 downto 0 );
        reverse          : out std_logic_vector( numberOfMotors - 1 downto 0 );
        enable           : out std_logic_vector( numberOfMotors - 1 downto 0 );
 
        -- Debug signals
        --
        displayMSN       : out std_logic_vector( 7 downto 0 );
        displayLSN       : out std_logic_vector( 7 downto 0 );

        res              : out std_logic;
        valid_out        : out std_logic_vector( numberOfMotors - 1 downto 0 )
      );

end controlbot; 

architecture RTL of controlbot is

  -- Use the data fusion engine for talking to all the external equipment
  --
  component data_fusion is

    generic( busWidth       : positive;
             numberOfMotors : positive 
           );

    port ( 
           -- Control signals
           --
           clock            : in  std_logic;
           reset            : in  std_logic;
           shaft_resolution : in  std_logic;

           -- Motor signals
           --
           pwm_signal       : in  std_logic_vector( numberOfMotors - 1 downto 0 );
           direction        : in  std_logic_vector( numberOfMotors - 1 downto 0 );
           stop             : in  std_logic_vector( numberOfMotors - 1 downto 0 );
           forward          : out std_logic_vector( numberOfMotors - 1 downto 0 );
           reverse          : out std_logic_vector( numberOfMotors - 1 downto 0 );
           enable           : out std_logic_vector( numberOfMotors - 1 downto 0 );

           -- Shaft encoder signals
           --
           reflection       : in  std_logic_vector( numberOfMotors - 1 downto 0 );
           velocity_valid   : out std_logic_vector( numberOfMotors - 1 downto 0 );
           velocity         : out std_logic_vector( numberOfMotors * busWidth - 1 downto 0 )
         );

  end component data_fusion;

  -- Use the motor control to control the motors to the reference velocity
  --
  component motor_control is

    generic ( busWidth         : positive; 

              -- Due to a compile bug with MaxPlus2, this must be specified to be
              -- busWidth + 1.
              --
              extendedBusWidth : positive;
              numberOfMotors   : positive;
              fuzzyInWidth     : positive;
              fuzzyOutWidth    : positive
            );

    port    ( clock            : in  std_logic;
              reset            : in  std_logic;
              velocity         : in  std_logic_vector( numberOfMotors * busWidth - 1 downto 0 );
              reference        : in  std_logic_vector( numberOfMotors * busWidth - 1 downto 0 );
              initial_rate     : in  std_logic_vector( numberOfMotors * busWidth - 1 downto 0 );
              diff_membership  : in  std_logic_vector( 7 * fuzzyInWidth - 1 downto 0 );
              int_membership   : in  std_logic_vector( 7 * fuzzyInWidth - 1 downto 0 );
              pwm_membership   : in  std_logic_vector( 7 * fuzzyOutWidth - 1 downto 0 );
              valid            : in  std_logic_vector( numberOfMotors - 1 downto 0 );
              pwm_signal       : out std_logic_vector( numberOfMotors - 1 downto 0 );
              fuzzy_out        : out std_logic_vector( 2 downto 0 );
              fuzzy_in         : out std_logic_vector( 2 downto 0 )
            );

  end component motor_control;

  component display is

    port ( clock  : in  std_logic;
           reset  : in  std_logic;
           valid  : in  std_logic;
           number : in  std_logic_vector( 3 downto 0 );
           output : out std_logic_vector( 7 downto 0 )
         );

  end component display;

  -- Define a milli-counter to control the motor updates
  --
  component milli_counter is

    generic ( busWidth        : positive;
              resolutionWidth : positive
            );

    port    ( clock           : in  std_logic;
              reset           : in  std_logic;
              resolution      : in  std_logic_vector( resolutionWidth - 1 downto 0 );
              count           : in  std_logic_vector( busWidth - 1 downto 0 );
              valid_out       : out std_logic
            );

  end component milli_counter;

  component frequency_divider is

    generic ( busWidth : positive );

    port    ( clock     : in  std_logic;
              reset     : in  std_logic;
              divisor   : in  std_logic_vector( busWidth - 1 downto 0 );
              frequency : out std_logic
            );

  end component frequency_divider;

  -- Define a reset signal so the board can be properly reset
  --
  signal reset              : std_logic;

  -- Define signals for connecting frequency divider to data fusion engine,
  -- and a frequency divider to the controller
  --
  signal shaft_resolution      : std_logic;
  signal shaft_frequency       : std_logic;
  signal milli_second_divisor  : std_logic_vector( resolutionWidth - 1 downto 0 );
  signal control_frequency     : std_logic_vector( busWidth - 1 downto 0 );

  -- Define signals for connection data fusion engine to motor control
  --
  signal pwm_signal         : std_logic_vector( numberOfMotors - 1 downto 0 );
  signal velocity           : std_logic_vector( numberOfMotors * busWidth - 1 downto 0 );

  -- Define reference signals
  --
  signal reference          : std_logic_vector( numberOfMotors * busWidth - 1 downto 0 );
  signal initial_rate       : std_logic_vector( numberOfMotors * busWidth - 1 downto 0 );

  -- Set both velocities valid every control resolution
  --
  signal valid              : std_logic_vector( numberOfMotors - 1 downto 0 );

  -- Create direction and stop signals for the motors
  --
  signal direction          : std_logic_vector( numberOfMotors - 1 downto 0 );
  signal stop               : std_logic_vector( numberOfMotors - 1 downto 0 );

  -- Membership functions to use for inputs
  --
  signal diff_membership    : std_logic_vector( 7 * fuzzyInWidth - 1 downto 0 );
  signal int_membership     : std_logic_vector( 7 * fuzzyInWidth - 1 downto 0 );

  -- Membership function to use for outputs
  --
  signal pwm_membership     : std_logic_vector( 7 * fuzzyOutWidth - 1 downto 0 );

  -- Define debug signals
  --
  signal fuzzy_out          : std_logic_vector( 3 downto 0 );
  signal fuzzy_in           : std_logic_vector( 3 downto 0 );

  signal one_signal         : std_logic;

begin

  -- Set-up debug signals
  --
  res          <= shaft_resolution;
  valid_out    <= valid;
  fuzzy_out(3) <= '0';
  fuzzy_in(3)  <= '0';
  one_signal   <= '1';
 
  -- Define reset line
  --
  reset <= reset_signal;

  -- Set the initial signals
  --
  reference    <= reference_velocity;
  initial_rate <= initial_duty_rate;

  -- Set up fuzzy membership functions
  --
  diff_membership <= fuzzy_difference;
  int_membership  <= fuzzy_int;
  pwm_membership  <= fuzzy_pwm;

  -- Set the divisor signal
  --
  milli_second_divisor   <= milli_second_resolution_divisor;
  control_frequency      <= control_update_frequency;

  mcount : frequency_divider generic map ( busWidth  => busWidth )
                             port    map (
                                           clock     => clock,
                                           reset     => reset,
                                           divisor   => milli_second_resolution_divisor,
                                           frequency => shaft_frequency
                                         );

  -- Create the shaft resolution
  --
  shaft : milli_counter generic map ( 
                                      busWidth        => busWidth,
                                      resolutionWidth => resolutionWidth
                                    )
                        port    map (
                                      clock           => clock,
                                      reset           => reset,
                                      resolution      => milli_second_divisor,
                                      count           => control_frequency,
                                      valid_out       => shaft_resolution
                                    );

  -- Map the data fusion engine
  --
  engine : data_fusion generic map ( 
                                     busWidth         => busWidth,
                                     numberOfMotors   => numberOfMotors
                                   )
                       port    map (
                                     -- Control signals
                                     --
                                     clock            => clock,
                                     reset            => reset,
                                     shaft_resolution => shaft_resolution,

                                     -- Motor Signals
                                     --
                                     pwm_signal       => pwm_signal,
                                     direction        => direction,
                                     stop             => stop,
                                     forward          => forward,
                                     reverse          => reverse,
                                     enable           => enable,

                                     -- Shaft encoder signals
                                     reflection       => reflection,
                                     velocity_valid   => valid,
                                     velocity         => velocity
                                   );

  -- Start the motor control module.  Control every ms.
  --
  control : motor_control generic map (
                                        busWidth         => busWidth,
                                        extendedBusWidth => extendedBusWidth,
                                        numberOfMotors   => numberOfMotors,
                                        fuzzyInWidth     => fuzzyInWidth,
                                        fuzzyOutWidth    => fuzzyOutWidth
                                      )
                          port    map (
                                        clock            => clock,
                                        reset            => reset,
                                        velocity         => velocity,
                                        reference        => reference,
                                        initial_rate     => initial_rate,
                                        diff_membership  => diff_membership,
                                        int_membership   => int_membership,
                                        pwm_membership   => pwm_membership,
                                        valid            => valid,
                                        pwm_signal       => pwm_signal,
                                        fuzzy_out        => fuzzy_out( 2 downto 0 ),
                                        fuzzy_in         => fuzzy_in( 2 downto 0 )
                                      );

  -- Show velocity(0) on the display
--  -- Show fuzzy output on display
  --
  display1 : display port map (
                                clock  => clock,
                                reset  => reset,
--                                valid  => valid(0),
                                valid  => one_signal,
                                number => fuzzy_out,
--                                number => velocity( 7 downto 4 ),
                                output => displayMSN
                              );

  -- Show velocity(0) on the display
  --
  display2 : display port map (
                                clock  => clock,
                                reset  => reset,
--                                valid  => valid(0),
                                valid  => one_signal,
                                number => fuzzy_in,
--                                number => velocity( 3 downto 0 ),
                                output => displayLSN
                              );

  -- Setup direction and stop signals
  --
  temporary : for i in 0 to numberOfMotors - 1 generate

    direction(i) <= '0';
    stop(i)      <= '0';

  end generate temporary;

end RTL;
