function c = nHopCDF(y,n,sigma,N)
% ================================================================================
% date      : 2009-03-16
% syntax    : nHopCDF(y,n,sigma,N)
% note      : computes the CDF sum of 'n' (zero mean) log-normal random variables
%             uses (34)-(36) of our Journal paper
% arguments : y     - return value "c" is P{Sum_n(Y_i) <= y}
%             n     - # hops
%             sigma - std. deviation of r.v. when in logarithmic form
%                   - a vector (of length 'n') 
%                   - if not: assumed the same for all variables
%             N     - # points for acceleration (should be odd)
% note      : assumes definition of MGF: M(s) = E{e^(-st)}
%             M(s) is the Laplace transform of the PDF
% ================================================================================

% error threshold for epsilon algorithm to stop
err = 1e-6; % for automatic execution

% validate parameters
n = n(1);
N = 2*floor(N(1)/2)+1; % 'N' should be odd
sigma = abs(sigma(:));
if length(sigma)~= n && length(sigma) ~= 1
   display(sprintf('nHopCDF: assuming %d variables, each having sigma = %d dB',n,10*sigma(1)/log(10)));
   sigma = sigma(1);
end

c = zeros(size(y));

for k=1:length(y) % repeat for each 'y'
    display(sprintf('nHopCDF: computing CDF for: y = %g (with N = %d)',y(k),N));
    c(k) = Accelerate(@(t)Ak(t, y(k), n, sigma), N, err);
end

% ================================================================================
% 'k'th term
function a = Ak(k, y, n, sigma)
a = 2/pi*quadl(@(t)realchf(t, y, n, sigma),k*pi+eps,(k+1)*pi,1e-6);

% the integrand
function chf = realchf(t, y, n, sigma)
chf = [];
for te = t(:)'
    temp = 1;
    for sigmaK = sigma(:)' % for each 'sigma'
        temp = temp * LNMGF(-j*te/y,sigmaK);
    end
    
    if length(sigma)==1
       temp = temp^n; 
    end
    
    chf = [chf; real(temp)/te*sin(te)];
end

% MGF function
function phi = LNMGF(s, sigma)
%phi = LNMGFConstPhaseInt6(s,sigma,10^-14);
phi = LNMGFConstPhaseLatest(s,sigma,10^-6); % to use newton rapson