function [mgf,npts] = LNMGFConstPhaseInt6(s,sigma,err,approach,z0)
% ================================================================================
% date      : 2008-07-06, 2008-08-18 (modified)
% syntax    : LNMGFConstPhaseInt5(s,sigma,err,z0)
% arguments : s     - argument of MGF (scalar)
%             sigma - std. deviation of the r.v. when in logarithmic form
%             err   - acceptable error rate
%             approach - 0 - mid point rule (default); 1 - trapezoidal rule; 2 - simpsons' rule
%             z0    - s*(sigma^2) may be externally passed to save invocation
%                     of maple here...
% note      : assumes M(s) = E{e^(-st)} double sided Laplace transform of the PDF
%           : provides a choice b/w Mid-point, Trapizoidala, Simpsons's rules
%           : uses single sided integration
% ================================================================================

% control parameters
debug = 0; % display debug info?

adjustRange = 1; % adjust range of integration
intRangePrecision = 10^-12; % precision used to truncate range of integration

% preprocess arguments
s = s(1);
sigma = sigma(1);

% make use of conjugate symmetry of M(s)
if imag(s)>0
    to_conjugate = 1;
    s = conj(s);
else
    to_conjugate = 0;
end
% note: imag(s) is -ve here...

if nargin < 3
    err = 10^(-9);
    approach = 0;
    z0 = -lambertw(s*(sigma^2)); % saddle point is not given (compute here)
elseif nargin < 4
    approach = 0;
    z0 = -lambertw(s*(sigma^2)); % saddle point is not given (compute here)
elseif nargin < 5
    z0 = -lambertw(s*(sigma^2)); % saddle point is not given (compute here)
else % needs to be +ve integer
    err = abs(err(1));
    if(err > 10^(-3))
        err = 10^(-3);
    end
    approach = round(abs(approach(1)));
end

if debug == 1
    display(sprintf('LNMGFConstPhaseInt6(%g + %g i,%g,%g,%g + %g i,%d)',real(s),imag(s),sigma,err,approach,real(z0),imag(z0))); % * * * debug * * *
end

% --------------------------------------------------
% parameters for internal use
zeta = s*(sigma^2); 
lambda = 1/(sigma^2); 
c = 1/sqrt(pi); 

x0 = real(z0); y0 = imag(z0)+eps;
c1 = y0*(x0-1); % const. phase

% --------------------------------------------------
prev = 0; cur = 0; % for looping
loopCount = 0;

den = sqrt(2*pi)*sigma; 
limit = 2*sqrt(2*(-log(intRangePrecision)-log(den))); % consider only range where normal PDF is above 'intRangePrecision'
limitHigh = sqrt(2)*sigma*limit+x0;

N = 8; % initial value; increase N till desired accuracy is obtained

while abs(prev-cur) > err || loopCount < 2

    if loopCount == 0 && adjustRange == 1
        % tighten the point of truncation (upper limit) 
        
        limitFixLoop = 1;
        while limitFixLoop < 50
            if abs(rt(limitHigh/1.1,sigma,zeta,z0,err))<intRangePrecision
               if debug == 1 % * * * debug * * *
                   display(sprintf('reducing upper limit from %g to %g',limitHigh,limitHigh/1.1));                    
               end
               limitHigh = limitHigh/1.1;
            else
                break;
            end
            
            limitFixLoop = limitFixLoop + 1;
        end
            
     end    
    
     % numerical integration (with N points)
     h = limitHigh/N;
    
     switch approach
     case 0 % mid-point rule
        t = h/2:h:limitHigh-h/2;
        mgf = (exp(-i*lambda*c1)*c*h)*sum(rt(t,sigma,zeta,z0,err)); % debug: displays intermediate values
             
     case 1 % trapezoidal rule
        if loopCount == 0 % first time
            t = [1:N-1]*h;
            mgf = (exp(-i*lambda*c1)*c*h)*sum(rt(t,sigma,zeta,z0,err));
            mgf = mgf + 0.5*(exp(-i*lambda*c1)*c*h)*(rt0(sigma,zeta,z0,err)+rt(limitHigh,sigma,zeta,z0,err)); % since lower limit is 0
        else % reuse points
            t = [1:2:N]*h; % only odd ones
            mgf = (exp(-i*lambda*c1)*c*h)*sum(rt(t,sigma,zeta,z0,err)) + mgf/2;
        end
        
     case 2 % Simpson's rule
         mgf = 2*sum(rt(2*h*[1:N/2-1],sigma,zeta,z0,err));
         mgf = mgf + 4*sum(rt(h*(2*[1:N/2]-1),sigma,zeta,z0,err));
         mgf = mgf + rt0(sigma,zeta,z0,err)+rt(limitHigh,sigma,zeta,z0,err); % since lower limit is 0
         mgf = mgf * (exp(-i*lambda*c1)*c*h)/3;
     case 3 % mid-point rule + Euler acceleration
        t = h/2:h:limitHigh-h/2;
        temp = rt(t,sigma,zeta,z0,err);
        Sn = (exp(-i*lambda*c1)*c*h)*cumsum(temp); % obtain the partial sums S_n^{(0)}
        %Sn(N) % result before acceleration
        % perform Euler approximation
        for k = 0:N-1 % compute S_n^{(k+1)}
            Sn(1:N-1) = 0.5*(Sn(1:N-1)+Sn(2:N));
        end
        mgf = Sn(1); % debug: displays intermediate values

     end 
    
    if to_conjugate == 1    
        mgf = conj(mgf);
    end

    if nargout == 2
        npts = N;
    end

    prev = cur; cur = mgf;
    loopCount = loopCount + 1;
    if debug == 1 % * * * debug * * *
        display(sprintf('loopCount = %d, N = %d, abs(prev-cur) = %g',loopCount,N, abs(prev-cur))); 
    end
    
    if loopCount > 12
        warning(sprintf('Failed converging with N = %d',N));
        break;
    end
    
    N = 2*N; % increase N    
end

% ----------------  ----------------------------------
% subfunctions
% --------------------------------------------------

% ------------------------------
% function : r(t)
function r = rt(t,sigma,zeta,z0,err)
r = qt(t,sigma,zeta,z0,err) + qt(-t,sigma,zeta,z0,err);

% ------------------------------
% function : r(0)
function r = rt0(sigma,zeta,z0,err)
r = 2*qt0(sigma,zeta,z0,err);

% ------------------------------
% function : q(t)
% the integrand, exp(-t^2)q(t) in (13)
% ------------------------------

function q = qt(t,sigma,zeta,z0,err)
theta = angle(zeta);
x0 = real(z0); y0 = imag(z0);
c1 = y0*(x0-1);
x = sqrt(2)*sigma*t + x0;
lambda = 1/(sigma^2); 
y = yfn(x,zeta,z0,c1,err); % compute 'y' on the contour for given 'x'
ReVz = abs(zeta)*exp(x).*cos(y+theta) + (x.^2-y.^2)/2;
q = exp(-lambda*ReVz).*(1+i*dydx(x,y,zeta,x0,y0));

% ------------------------------
% function : q(0)
% ------------------------------

function q = qt0(sigma,zeta,z0,err)
theta = angle(zeta);
x0 = real(z0); y0 = imag(z0);
lambda = 1/(sigma^2); 
ReVz = abs(zeta)*exp(x0)*cos(y0+theta) + (x0^2-y0^2)/2;
q = exp(-lambda*ReVz)*(1+i*(y0/(sqrt(y0^2+(1-x0)^2)+(1-x0))));

% ------------------------------
% function : dydx
% gradient (dy/dx) along the contour, given 'x' & 'y'
% 'x' ,'y' are vectors of same length; 'zeta','x0','y0' is a scalar
% ------------------------------

function d = dydx(x,y,zeta,x0,y0)
d = -(abs(zeta)*exp(x).*sin(y+angle(zeta))+y)./(abs(zeta)*exp(x).*cos(y+angle(zeta))+x);
sp = isnan(d); % NAN arises at saddle point
if any(sp)  
   display('Warning: NAN computed in the gradient corresponding to the saddle point');
end
d(sp) = y0/(sqrt(y0^2+(1-x0)^2)+(1-x0));

% ------------------------------
% function : yfn
% x is possibly a vector, other arguments are scalars
% ------------------------------

function y = yfn(x,zeta,z0,c1,err)
y = zeros(size(x));

if isreal(z0) % contour is the real axis!!!
    return;
end
theta = angle(zeta);

for k=1:length(x) % for each 'x'
    
    if x(k) < real(z0)
        yleft = 0; yright = imag(z0);
    else
        yleft = imag(z0); yright = -theta;
    end
        
    if fy(yleft,x(k),zeta,c1)*fy(yright,x(k),zeta,c1) > 0
        warning(sprintf('left & right bounds (%g,%g) yield the same sign (for x=%g)',yleft,yright,x(k)));
        ymid = (yleft + yright)/2; y(k) = ymid; % assume mid point...
        break;
    end
    
    loop = 0;
    while loop>=0
        ymid = (yleft + yright)/2;
        if fy(yleft,x(k),zeta,c1)*fy(ymid,x(k),zeta,c1) > 0
            yleft = ymid;
        else
            yright = ymid;
        end
        loop = loop + 1;
        if abs(ymid-(yleft+yright)/2) < err/10 % convergence (work at higher precision)
           y(k) = ymid; 
           loop = -1;
        end
        if loop > 100
            warning(sprintf('could not find ''y'' after looping %d times for ''x'' = %g',loop,x(k)));
            loop = -1;
        end
    end
end

% --------------------------------------------------
% function : f(y)
% used by the bisection method
% 'c1','zeta' are possibly vectors
% --------------------------------------------------
function f = fy(y,x,zeta,c1)
f = abs(zeta)*exp(x).*sin(y+angle(zeta))+x*y-c1;