function y = Accelerate(fnHandle, N, err)
% ================================================================================
% date      : 2009-03-14
% syntax    : y = Accelerate(fnHandle, N, err)
% note      : performs convergence acceleration (using epsilon method)
% arguments : fnHandle - handle of a function that takes in an integer;
%                        computes 'k'th term (starting k = 0)
%             N        - starting sequence length for acceleration;
%                      - N should be ODD; it is increased till convergence
%             err      - error tolerence (used to determine convergence)
% ================================================================================

N = abs(N(1));
if N > 100
   error('Accelerate: ''N'' should be smaller than 100');
else
   N = 2*floor(N/2)+1; % force 'N' to be ODD
end
 
% innitial values
k = 0; 
s = fnHandle(0); % 0th term

while(k >= 0)
    if N > 100
       error('Accelerate: convergence acceleration failed (reached N = 100)');
    end
    
    % -------------------------    
    % compute the terms 'k+1' to 'N'
    % - terms are reused if N is increased in the outer loop
    % -------------------------
    for k=k+1:N
        term = fnHandle(k);
        s = [s; term];

        %if abs(term) < 1e-15 && sum(s)>0
        %    N = k; break;
        %end
    end

    % -------------------------    
    % convergence acceleration
    % -------------------------
    x = cumsum(s); % partial sums

    % compute the epsilon array
    E = zeros(N,N+1);
    for m=1:N,
        E(m,2) = x(m);
    end

    for m=3:N+1,
        for n=1:N+2-m,
           E(n,m) = E(n+1,m-2) + 1 /( E(n+1,m-1) - E(n,m-1));
        end
    end

    sa = [];
    for m = 1:(N+1)/2,
        sa = [sa; E(N-2*m+2,2*m)];
    end
    sa = sa(~isnan(sa)); % ignore NAN terms (if any)

    % -------------------------    
    % check for convergence...
    % -------------------------
    difference = [sa; 0] - [0; sa];
    difference = difference(1:length(sa));

    m = min(find(abs(difference)<err));
    if length(m(:)) < 1
       % has not converged yet; retry with increased N
       N = N + 2;
       %display(sprintf('Accelerate: increasing N to %d',N));
    else
       % converged!
       y = sa(m);
       break;
    end
end

